/*
Code to replicate tables and figures in:

Luke N. Condra and Jacob N. Shapiro, "Who Takes the Blame? The Strategic Effects of Collateral Damage,
American Journal of Political Science 56, no. 1 (January 2012): 167-87.

Corresponding author: Jacob N. Shapiro (jns@princeton.edu)
Revised: August 2, 2012
Input files: cs_repdata_2012.dta
Supporting Evidence and data available from: http://www.princeton.edu/~jns
*/

clear
clear mata
* set directory pathway here
set more off
set mem 1g


***** MAIN TABLES *****

* Table 1. Predicting population-weighted SIGACTs per week

use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate , cluster(district) a(sect_half)


* Table 1 test of exogeneity
qui areg d.p_S1 ld.co_min l.co_min pop_den urate , cluster(district) a(sect_half)
test l.co_min
qui areg d.p_S1 ld.in_min l.in_min pop_den urate , cluster(district) a(sect_half)
test l.in_min
qui areg d.p_S1 ld.sec_min l.sec_min pop_den urate , cluster(district) a(sect_half)
test l.sec_min
qui areg d.p_S1 ld.unk_min l.unk_min pop_den urate , cluster(district) a(sect_half)
test l.unk_min 
qui areg d.p_S1 ld.co_min ld.in_min l.co_min l.in_min pop_den urate , cluster(district) a(sect_half)
test l.co_min 
test l.in_min

* Table 2. Predict First Difference of SIGACTs per Week as Function of Civilian Casualties

areg d.p_S1 ld.co_min ld.in_min pop_den urate, cluster(district) absorb(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==4, cluster(district) absorb(half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==2, cluster(district) absorb(half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==3, cluster(district) absorb(half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==1, cluster(district) absorb(half)

* For Table 3 see matching section below.

***** SUPPLEMENTARY TABLES *****

/*
SE Table 1A
Proportion unknown as function of violence. Show % unattributable uncorrelated with incidents 
as soon as put in any kind of regional controls. With full set of time and space FE correlation goes to 0.
*/

use cs_repdata_2012.dta, clear

reg unk_ev p_S1 , cluster(district)
xi: reg unk_ev p_S1 i.sect, cluster(district)
areg unk_ev p_S1 , cluster(district) a(half)
areg unk_ev p_S1, cluster(district) absorb(district) 
areg unk_ev p_S1 h_* , cluster(district) absorb(district)

reg unk_k p_S1, cluster(district)
xi: reg unk_k p_S1 i.sect, cluster(district)
areg unk_k p_S1 , cluster(district) a(half)
areg unk_k p_S1, cluster(district) absorb(district) 
areg unk_k p_S1 h_* , cluster(district) absorb(district)

/* 
SE Table 1B: Proportion not attributed at district level by governorate-week as function of governorate level violence to test for reporting bias.
*/

use civcas_table_se1b.dta, clear
xtset governorate_1 week

reg prop_disknown p_S1 , cluster(governorate)
xi: reg prop_disknown p_S1 i.sect, cluster(governorate)
areg prop_disknown p_S1 , cluster(governorate) a(sh)

reg prop_coalition_disknown p_S1 , cluster(governorate)
xi: reg prop_coalition_disknown p_S1 i.sect, cluster(governorate)
areg prop_coalition_disknown p_S1 , cluster(governorate) a(sh)

reg prop_insurgent_disknown p_S1 , cluster(governorate)
xi: reg prop_insurgent_disknown p_S1 i.sect, cluster(governorate)
areg prop_insurgent_disknown p_S1 , cluster(governorate) a(sh)

*SE Table 1C. Descriptive statistics
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

sum SIGACT SIG_1 event min coalition co_min insurgent in_min sectarian sec_min unknown unk_min
bysort sect: sum SIGACT SIG_1 event min coalition co_min insurgent in_min sectarian sec_min unknown unk_min

*SE Table 2A. Core results controlling for pre-existing trend and district FE
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min ld.in_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate ld.p_S1, cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate ld.p_S1 sh_2-sh_44, cluster(district) a(district)

*SE Table 2B. Core regressions dropping Baghdad
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min pop_den urate if governorate != "Baghdad", cluster(district) a(sect_half)
areg d.p_S1 ld.in_min pop_den urate if governorate != "Baghdad", cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min pop_den urate if governorate != "Baghdad", cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if governorate != "Baghdad", cluster(district) a(sect_half)

*SE Table 2C. Predict first difference of SIGACTs per week as function of civilian casualties (placebo test)
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 fd.co_min fd.in_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 fd.co_min fd.in_min pop_den urate if sect==4, cluster(district) a(sect_half)
areg d.p_S1 fd.co_min fd.in_min pop_den urate if sect==2, cluster(district) a(sect_half)
areg d.p_S1 fd.co_min fd.in_min pop_den urate if sect==3, cluster(district) a(sect_half)
areg d.p_S1 fd.co_min fd.in_min pop_den urate if sect==1, cluster(district) a(sect_half)

/*
SE Table 2D. Predict civilian casualties with leads of SIGACTs/100,000 per week
Reversing the regression shows that violence does not predict civilian killings 
for Coalition or insurgents once we estimate the model in first difference.
*/
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg co_min f.p_S1 pop_den urate , cluster(district) a(sect_half)
areg d.co_min fd.p_S1 pop_den urate , cluster(district) a(sect_half)

areg in_min f.p_S1 pop_den urate , cluster(district) a(sect_half)
areg d.in_min fd.p_S1 pop_den urate , cluster(district) a(sect_half)

areg sec_min f.p_S1 pop_den urate , cluster(district) a(sect_half)
areg d.sec_min fd.p_S1 pop_den urate , cluster(district) a(sect_half)

*SE Table 2E. Effect of civilian casualties on average levels of violence over next month
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d4map_S1 ld.co_min ld.in_min , cluster(district) absorb(sect_half)
areg d4map_S1 ld.co_min ld.in_min if sect==1 , cluster(district) absorb(sect_half)
areg d4map_S1 ld.co_min ld.in_min if sect==2 , cluster(district) absorb(sect_half)
areg d4map_S1 ld.co_min ld.in_min if sect==3 , cluster(district) absorb(sect_half)
areg d4map_S1 ld.co_min ld.in_min if sect==4 , cluster(district) absorb(sect_half)

*SE Table 2F. Effect of Civilian Casualties by Urbanity
*gen urban = /* median */
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min ld.in_min if pct_urban > .4885, cluster(district) absorb(sect_half)
areg d.p_S1 ld.co_min ld.in_min if pct_urban <= .4885, cluster(district) absorb(sect_half)
areg d.p_S1 ld.co_min ld.in_min ld.uco ld.uin, cluster(district) absorb(sect_half)

	*F-Test statistics
areg d.p_S1 ld.co_min ld.in_min ld.uco ld.uin, cluster(district) absorb(sect_half)
test ld.uco=ld.co_min
test ld.uin=ld.in_min
test ld.uin ld.in_min
test ld.uin ld.uco

*SE Table 2G. Regressing attacks by type (IEDs, direct fire, indirect fire, and suicide attacks) on civilian casualties
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min ld.in_min , cluster(district) absorb(sect_half)
areg d.p_df ld.co_min ld.in_min, cluster(district) absorb(sect_half)
areg d.p_idf ld.co_min ld.in_min, cluster(district) absorb(sect_half)
areg d.p_ied ld.co_min ld.in_min, cluster(district) absorb(sect_half)
areg d.p_suicide ld.co_min ld.in_min, cluster(district) absorb(sect_half)

*SE Table 2H. Core Regressions Weighted by District Population
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min pop_den urate pop_den_co_min [aw=POP], cluster(district) a(sect_half)
areg d.p_S1 ld.in_min pop_den urate pop_den_co_min [aw=POP], cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min pop_den urate pop_den_co_min [aw=POP], cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min pop_den urate pop_den_co_min [aw=POP], cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate pop_den_co_min [aw=POP], cluster(district) a(sect_half)

/*SE Table 2I. Core regressions with log of civilian casualties on RHS
Logged value set to -1 where 0 casualties, so that difference from 0 to 1 is close to difference from 1 to 2. */
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min_ln pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min_ln pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min_ln pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min_ln pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min_ln ld.in_min_ln pop_den urate , cluster(district) a(sect_half)

*SE Table 2J: Core regressions with count of incidents by each party on the RHS
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min ld.coalition pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min ld.insurgent pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min ld.sectarian pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min ld.unknown pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.coalition ld.in_min ld.insurgent pop_den urate , cluster(district) a(sect_half)

*SE Table 2K: Core regressions with indicator variable for non-zero civilian casualties on the RHS
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min l.co_kill pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min l.in_kill pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min l.sec_kill pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min l.unk_kill pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min l.co_kill ld.in_min l.in_kill pop_den urate , cluster(district) a(sect_half)

*SE Table 2L: Core regressions with spatial lag of SIGACTs on RHS
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min d.SIG_1_sl pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min d.SIG_1_sl pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min d.SIG_1_sl pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min d.SIG_1_sl pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min d.SIG_1_sl pop_den urate , cluster(district) a(sect_half)

*SE Table 2M: Core regressions with dummy for weeks that includes first day of month
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min mi pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.in_min mi pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min mi pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min mi pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min mi pop_den urate , cluster(district) a(sect_half)

/*SE Table 2N. Core regressions dropping IBC incidents where both insurgents and Coalition involved (7.6% of incidents)
Without double-counting, FD results by sect*/
use civcas_table_se2n.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min pop_den urate, cluster(district) a(sect_half)
areg d.p_S1 ld.in_min pop_den urate  , cluster(district) a(sect_half)
areg d.p_S1 ld.sec_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.unk_min pop_den urate , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate , cluster(district) a(sect_half)

/*SE Table 2O. Core regressions dropping IBC incidents where both insurgents and Coalition involved (7.6% of incidents)
Without double-counting, FD results by sect*/
use civcas_table_se2n.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.co_min ld.in_min pop_den urate, cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==4 , cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==2, cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==3, cluster(district) a(sect_half)
areg d.p_S1 ld.co_min ld.in_min pop_den urate if sect==1, cluster(district) a(sect_half)

/*SE Table 2P. Core regressions with population-weighted civilian casualties on RHS
Clean regressions by sect with population weighting */
use cs_repdata_2012.dta, clear
xtset district_id week, weekly

areg d.p_S1 ld.p_co_min ld.p_in_min pop_den urate, cluster(district) absorb(sect_half)
areg d.p_S1 ld.p_co_min ld.p_in_min pop_den urate if sect==4, cluster(district) absorb(half)
areg d.p_S1 ld.p_co_min ld.p_in_min pop_den urate if sect==2, cluster(district) absorb(half)
areg d.p_S1 ld.p_co_min ld.p_in_min pop_den urate if sect==3, cluster(district) absorb(half)
areg d.p_S1 ld.p_co_min ld.p_in_min pop_den urate if sect==1, cluster(district) absorb(half)

*SE Table 2Q. Core regressions using only matched district weeks

areg dp_S1 ld_co_min ld_in_min pop_den urate if cem_matched==1, cluster(district) a(sect_half)
areg dp_S1 ld_co_min ld_in_min pop_den urate if cem_matched==1 & sect==4, cluster(district) a(sect_half)
areg dp_S1 ld_co_min ld_in_min pop_den urate if cem_matched==1 & sect==2, cluster(district) a(sect_half)
areg dp_S1 ld_co_min ld_in_min pop_den urate if cem_matched==1 & sect==3, cluster(district) a(sect_half)
areg dp_S1 ld_co_min ld_in_min pop_den urate if cem_matched==1 & sect==1, cluster(district) a(sect_half)

*SE Table 3. Impact of population density and urbanity on civilian casualty ratios

areg min_rat1 pct_urban pop_den q_* , absorb(district) cluster(district)
test pct_urban pop_den
areg co_min_rat1 pct_urban pop_den q_* , absorb(district) cluster(district)
test pct_urban pop_den
areg in_min_rat1 pct_urban pop_den q_* , absorb(district) cluster(district)
test pct_urban pop_den
areg sec_min_rat1 pct_urban pop_den q_* , absorb(district) cluster(district)
test pct_urban pop_den
areg unk_min_rat1 pct_urban pop_den q_* , absorb(district) cluster(district)
test pct_urban pop_den

*SE Table 4. Impact of CERP projects and spending on civilian casualty ratios
use civcas_table_se4.dta, clear

areg min_rat1 cerp_start cerp_total q_* , absorb(district) cluster(district)
test cerp_start cerp_total
areg co_min_rat1 cerp_start cerp_total q_* , absorb(district) cluster(district)
test cerp_start cerp_total
areg in_min_rat1 cerp_start cerp_total q_* , absorb(district) cluster(district)
test cerp_start cerp_total
areg sec_min_rat1 cerp_start cerp_total q_* , absorb(district) cluster(district)
test cerp_start cerp_total
areg unk_min_rat1 cerp_start cerp_total q_* , absorb(district) cluster(district)
test cerp_start cerp_total

*SE Table 5. Estimated change in SIGACTs (per 100,000 people) from increasing civilian casualties by 1 SD

* Do the multiplication.....

***** FIGURES *****

* Fig. 1. Insurgent attacks and civilian casualties in Iraq by party responsible (Feb 2004 - Feb 2009)
use cs_repdata_countrymonth_2012.dta, clear

format month %tm
tsset month

twoway (line co_min month , lcolor(cranberry) lpattern(longdash_dot) yaxis(1)) ///
	(line in_min month , lcolor(sienna) lpattern(dot) yaxis(1)) (line sec_min month , lcolor(orange) ///
	lpattern(dash) yaxis(1)) (line unk_min month , lcolor(maroon) lpattern(vshortdash) yaxis(1)) ///
	(line SIG_1 month , lcolor(blue) yaxis(2)), ytitle(Civilian Casualties (by type), axis(1)) ///
	ytitle(Total Insurgent Attacks, axis(2)) ytitle(, size(small) axis(1)) ytitle(, size(small) ///
	axis(2)) ylabel(0(500)3500, labsize(vsmall) axis(1)) ylabel(, labsize(vsmall) axis(2)) ///
	xtitle(Month) xtitle(, size(small)) xlabel(, labsize(vsmall)) xmtick(##2) ///
	title(Fig. 1. Insurgent Attacks and Civilian Casualties in Iraq by Party Responsible, size(small)) ///
	subtitle((Feb. 2004 - Feb. 2009), size(vsmall)) ///
	legend(on order(1 "Coalition CIVCAS" 2 "Insurgent CIVCAS" 3 "Sectarian CIVCAS" 4 "Unknown CIVCAS" 5 "Total Insurgent Attacks") ///
	row(2) rowgap(tiny) colgap(tiny) size(vsmall))


* Fig. 2. Insurgent attacks in Al Sadr and Karkh (Feb 2004 - Feb 2009)
use cs_repdata_2012.dta, clear
keep if district_id==8|district_id==69
save al-sadr_karkh.dta, replace

xtline SIG_1, addplot(lowess SIG_1 week) ytitle(Total Insurgent Attacks) ytitle(, size(small)) ttitle(Week) ///
	xlabel(, labsize(vsmall)) ttitle(, size(small)) byopts(title(Fig. 2. Insurgent Attacks in Al Sadr and Karkh, size(small))  ///
	subtitle((Feb. 2004 - Feb. 2009), size(vsmall))) legend(label(1 "Sum of SIGACTS in district-week") label(2 "Lowess Curve") size(small))


* Fig 3. Civilian casualties in Al-Muqdadiya and Al-Musayab (Feb 2004 - Feb 2009)
use cs_repdata_2012.dta, clear
keep if district_id==25|district_id==26

xtline min, addplot(lowess min week) ytitle(Total Civilian Casualties) ///
	ytitle(, size(small)) xlabel(, labsize(vsmall)) ttitle(Week) ttitle(, size(small)) ///
	byopts(title(Fig. 3. Civilian Casualties in Al-Muqdadiya and Al-Musayab, size(small)) ///
	subtitle((Feb. 2004 - Feb. 2009), size(vsmall))) legend(label(1 "Civilian Casualties") label(2 "Lowess Curve") size(small))


* Fig. 4. Insurgent attacks and civilian casualties in Baghdad districts (Feb 2004 - Feb 2009)

use cs_repdata_2012.dta, clear

keep if governorate=="Baghdad"
xtset district_id week, weekly
	
gen civcas = in_min + co_min

twoway (tsline civcas) (tsline SIG_1, yaxis(2)), ytitle(Coalition- and Insurgent-caused Civilian Casualties) ///
	ytitle(, size(small)) ylabel(, labsize(vsmall)) ytitle(Total Insurgent Attacks, axis(2)) ///
	ytitle(, size(small) orientation(vertical) axis(2)) ylabel(, labsize(vsmall) axis(2)) ///
	ttitle(Week) ttitle(, size(small)) tlabel(, labsize(vsmall)) by(, title(Fig. 4. Insurgent Attacks and Civilian Casualties in Baghdad Districts, size(small)) ///
	subtitle((Feb. 2004 - Feb. 2009), size(vsmall))) legend(order(1 "Civilian Casualties (left axis)" 2 "Insurgent Attacks (right axis)") rows(1) size(vsmall)) by(district)


* Figure 5, see matching code below.
	
	
* APPENDIX FIGURES

* SE Fig 1. Population-weighted insurgent attacks in Iraq by district
use cs_repdata_2012.dta, clear
sort district week
xtset district_id week, weekly

xtline p_S1, ytitle(Population-Weighted Insurgent Attacks) ytitle(, size(small)) ttitle(Week) ///
	ttitle(, size(small)) byopts(title(Fig. 1. Population-Weighted Insurgent Attacks in Iraq by District, size(small)) subtitle((Feb. 2004 - Feb. 2009), size(vsmall)))


* SE Fig 2. Civilian casualties (per 100,000) in Iraq by district

xtline min, ytitle(Civilian Casualties per 100000 people) ytitle(, size(small)) ttitle(Week) ///
	ttitle(, size(small)) byopts(title(Fig. 2. Civilian Casualties (per 100,000) in Iraq by District, size(small)) subtitle((Feb. 2004 - Feb. 2009), size(vsmall)))


* SE Fig. 3. Sectarian violence by sect-region

collapse (sum) SIGACT SIG_1 event coalition sectarian insurgent unknown min co_min in_min sec_min unk_min, by(sect week)

	*Drop single outlier in Mixed area
drop if sec_min==1004

twoway (tsline sec_min), ytitle(Sectarian Violence Casualties) ytitle(, size(small)) ///
	ttitle(Week) ttitle(, size(small)) tlabel(minmax, labsize(vsmall)) by(, title(Fig. 3. Sectarian Violence Casualties by Sect Region, size(small))) by(sect)


* Finally matching tables *********************************


* Table 3. Matching estimate of impact of Coalition or insurgent killings in period (t) on SIGACTs/100,000 at period (t+1)
use cs_repdata_2012, clear
xtset district_id week, weekly

qui tab year, gen(y_)
* Fix 2009 so it gets same FE as 2008
xtset district_id week, weekly
gen half1= half-87
gen half2 = half1^2
gen half3 = half1^3
gen q1 = quarter-175
gen q2 = q1^2
gen q3 = q1^3
gen month1=month-527
gen month2=month1^2
gen month3=month1^3


* Generate lags and leads of incidents
local vars "p_S1"
foreach y of local vars {
	qui forval x = 1/7 {
		gen l`x'`y' = l`x'.`y'
		gen f`x'`y' = f`x'.`y'
		}
	}

* Generate quantiles for each district/week on each of the violence measures and then produce

*local vars "S1 df ied ied_cleared totied"
local vars "S1"
foreach x of local vars {
	centile lma4_p`x' if lma4_p`x' > 0, centile(10 33 66 90)  
	centile lma4_p`x' if lma4_p`x' > 0, centile(10 30 50 70 90)    
	}


drop l0dp l1dp l2dp l3dp l4dp
xtset district_id week, weekly

* +/- 1 is the 25th and 75th percentile of data for district weeks with changes. 
gen l0dp = (d.p_S1 > 1)
replace l0dp = -1 if d.p_S1 < -1
forval x = 1/4 {
	gen l`x'dp = (l`x'd.p_S1 > 1)
	replace l`x'dp = -1 if l`x'd.p_S1 < -1
	}
		
*imbalance is .666
imbalance lma4_pS1 l0dp l1dp l2dp l3dp l4dp, tr(co_in_kill)

* Uses 4-week MA of total attacks at 10 33 66 90, current and 4 lags of differences pos/0/neg. 
* There are 241 combinations of the differences which is just a bit less than 3^5=243
* Looks really good, L1 of .4183
drop cem_strata cem_matched cem_weights cem_w
cem lma4_pS1 (.129 .523 2.437 11.13) l0dp (-.5 .5) l1dp (-.5 .5) l2dp (-.5 .5)  l3dp (-.5 .5) l4dp (-.5 .5) , tr(co_in_kill) show 

by cem_strata, sort : egen float cem_w = count(cem_strata)

drop if cem_w < 3
drop if tot_p_S1 == 0 & tot_co_min==0 &  tot_in_min==0 

* Collect regression coefficients, dropping strata on which it cannot be estimated
xtset district_id week, weekly
local lago "f7p f6p f5p f4p f3p f2p f1p p l1p l2p l3p l4p l5p l6p l7p"

foreach x of local lago {
	qui statsby _b _se, by(cem_strata)  saving(coin_kill_betas_`x'.dta, replace): reg `x'_S1 co_kill in_kill q1 q2 q3
	qui statsby _b _se, by(cem_strata)  saving(coin_min_betas_`x'.dta, replace): reg `x'_S1 co_min in_min q1 q2 q3
 
	merge cem_strata using coin_kill_betas_`x'.dta, uniqusing sort
	drop _b_cons _se_cons _merge

	merge cem_strata using coin_min_betas_`x'.dta, uniqusing sort
	drop _b_cons _se_cons _merge
	
	rename _b_co_kill _b_co_kill_`x'
	rename _se_co_kill _se_co_kill_`x'
	rename _b_in_kill _b_in_kill_`x'
	rename _se_in_kill _se_in_kill_`x'
	rename _b_co_min _b_co_min_`x' 
	rename _se_co_min _se_co_min_`x'
	rename _b_in_min _b_in_min_`x'
	rename _se_in_min _se_in_min_`x'
	}

sort cem_strata week district
gen marker = 1 if cem_strata[_n] != cem_strata[_n+1]
drop if marker != 1
drop if cem_w>800

* Then do results
mean _b_co_kill_f1p [iw=cem_w] if marker==1 & cem_w < 300
mean _b_co_min_f1p [iw=cem_w] if marker==1 & cem_w < 300
mean _b_in_kill_f1p [iw=cem_w] if marker==1 & cem_w < 300
mean _b_in_min_f1p [iw=cem_w] if marker==1 & cem_w < 300

* Mixed
mean _b_co_kill_f1p [iw=cem_w] if marker==1 & sect == 2 & cem_w < 300
mean _b_co_min_f1p [iw=cem_w] if marker==1 & sect == 2 & cem_w < 300
mean _b_in_kill_f1p [iw=cem_w] if marker==1 & sect == 2 & cem_w < 300
mean _b_in_min_f1p [iw=cem_w] if marker==1 & sect == 2 & cem_w < 300

* Sunni
mean _b_co_kill_f1p [iw=cem_w] if marker==1 & sect == 4 & cem_w < 300
mean _b_co_min_f1p [iw=cem_w] if marker==1 & sect == 4 & cem_w < 300
mean _b_in_kill_f1p [iw=cem_w] if marker==1 & sect == 4 & cem_w < 300
mean _b_in_min_f1p [iw=cem_w] if marker==1 & sect == 4 & cem_w < 300

expand = 15

* Generate variable that identifies lag
bysort cem_strata week: gen lago = [_n]
replace lago = lago - 8

* Turn _b_* variables into one that gives coefficient and se for insurgent and coalition at each lag
local wh1 "co_kill co_min in_kill in_min"
local what "b se"
local lags1 "1 2 3 4 5 6 7"
foreach y of local what {
	foreach x of local wh1 {
		foreach z of local lags1 {
			qui replace _`y'_`x'_p = _b_`x'_f`z'p if lago == `z'
			qui replace _`y'_`x'_p = _b_`x'_l`z'p if lago == -`z'
			}
		}
	}


* Generate some additional variables for the graph
* Use loop to generate graphing variables for differences and treatment and control
local wh1 "co_kill co_min in_kill in_min"
qui foreach z of local wh1 {
	sort cem_strata lago
	gen mean_`z' = 0 
	gen se_`z' = 0 
	gen a_`z' = 0 
	gen upper_`z' = 0 
	gen lower_`z' = 0 
	
	forval x = -7/7 {
		mean _b_`z'_p if lago == `x' & cem_w < 300 [iw=cem_w]
		matrix b1=e(b) 
		matrix V1=e(V)
		scalar b1=b1[1,1] 
		scalar varb1=V1[1,1] 
		scalar list b1 varb1
	
		replace mean_`z' = b1 if lago == `x'
		replace se_`z' = sqrt(varb1) if lago == `x'
		replace a_`z' = 1.96 * se_`z' if lago == `x'
		replace upper_`z' = mean_`z' + a_`z' if lago == `x'
		replace lower_`z' = mean_`z' - a_`z' if lago == `x'
		}
	}


sort cem_strata lago

twoway (line mean_co_min lago if lago > -6 & lago < 8 & cem_strata==1 , lcolor(blue) xline(0, lcolor(black)) ) ///
	(line upper_co_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)) ///
	(line lower_co_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)), ///
	title("Marginal Effect of Coalition Killings", size(medsmall)) ylab(-.5(.5)1) ///
	ytitle(SIGACTS/100000, size(medsmall)) ytitle(, linegap(.2)) ylabel(, glcolor(gs12)) ///
	xtitle(Period from treatment) xtitle(, size(medsmall)) xtitle(, margin(small)) xtick(-5(1)7) xlabel(-5(1)7) ///
	legend(col(2) order(1 2) label(1 "Weighted difference in means") size(vsmall)  ///
	label(2 "95% Confidence Interval") label(3 " ") off) name(dif_co_min, replace) 

twoway (line mean_in_min lago if lago > -6 & lago < 8 & cem_strata==1 , lcolor(blue) xline(0, lcolor(black)) ) ///
	(line upper_in_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)) ///
	(line lower_in_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)), ///
	title("Marginal Effect of Insurgent Killings", size(medsmall)) ///
	ytitle(SIGACTS/100000, size(medsmall)) ytitle(, linegap(.2)) ylabel(, glcolor(gs12)) ///
	xtitle(Period from treatment) xtitle(, size(medsmall)) xtitle(, margin(small)) xtick(-5(1)7) xlabel(-5(1)7) ///	
	legend(col(2) order(1 2) label(1 "Weighted difference in means") size(vsmall) ///
	label(2 "95% Confidence Interval") label(3 " ") off) name(dif_in_min, replace) 
		
graph combine dif_co_min dif_in_min, colfirst cols(1) ///
	title(Figure 3. Effect of Civilian Killings on Insurgent Violence in Matched Strata, size(medsmall)) ///
	subtitle((Matching on 4-period history of SIGACTS/100000, and average insurgent attacks), size(small)) imargin(small) ///
	name(cem_pooled_q3, replace)

** Appendix figure SE 5 - Tight history match

use cs_repdata_2012, clear
xtset district_id week, weekly

qui tab year, gen(y_)
* Fix 2009 so it gets same FE as 2008
xtset district_id week, weekly
gen half1= half-87
gen half2 = half1^2
gen half3 = half1^3
gen q1 = quarter-175
gen q2 = q1^2
gen q3 = q1^3
gen month1=month-527
gen month2=month1^2
gen month3=month1^3


* Generate lags and leads of incidents
local vars "p_S1"
foreach y of local vars {
	qui forval x = 1/7 {
		gen l`x'`y' = l`x'.`y'
		gen f`x'`y' = f`x'.`y'
		}
	}

* Generate quantiles for each district/week on each of the violence measures and then produce

*local vars "S1 df ied ied_cleared totied"
local vars "S1"
foreach x of local vars {
	centile lma4_p`x' if lma4_p`x' > 0, centile(10 33 66 90)  
	centile lma4_p`x' if lma4_p`x' > 0, centile(10 30 50 70 90)    
	}


drop l0dp l1dp l2dp l3dp l4dp
xtset district_id week, weekly

* +/- 1 is the 25th and 75th percentile of data for district weeks with changes. 
gen l0dp = (d.p_S1 > .5)
replace l0dp = -1 if d.p_S1 < -.5
forval x = 1/4 {
	gen l`x'dp = (l`x'd.p_S1 > .5)
	replace l`x'dp = -1 if l`x'd.p_S1 < -.5
	}
		
*imbalance is .6700
imbalance lma4_pS1 l0dp l1dp l2dp l3dp l4dp, tr(co_in_kill)

* Uses 4-week MA of total attacks at 10 33 66 90, current and 4 lags of differences pos/0/neg. 
* There are 241 combinations of the differences which is just a bit less than 3^5=243
* Looks really good, L1 of .4189 
drop cem_strata cem_matched cem_weights cem_w
cem lma4_pS1 (.129 .523 2.437 11.13) l0dp (-.5 .5) l1dp (-.5 .5) l2dp (-.5 .5)  l3dp (-.5 .5) l4dp (-.5 .5) , tr(co_in_kill) show 

by cem_strata, sort : egen float cem_w = count(cem_strata)

drop if cem_w < 3
drop if tot_p_S1 == 0 & tot_co_min==0 &  tot_in_min==0 

* Collect regression coefficients, dropping strata on which it cannot be estimated, this can be slow.
xtset district_id week, weekly
local lago "f7p f6p f5p f4p f3p f2p f1p p l1p l2p l3p l4p l5p l6p l7p"

foreach x of local lago {
	qui statsby _b _se, by(cem_strata)  saving(coin_kill_betas_`x'.dta, replace): reg `x'_S1 co_kill in_kill q1 q2 q3
	qui statsby _b _se, by(cem_strata)  saving(coin_min_betas_`x'.dta, replace): reg `x'_S1 co_min in_min q1 q2 q3
 
	merge cem_strata using coin_kill_betas_`x'.dta, uniqusing sort
	drop _b_cons _se_cons _merge

	merge cem_strata using coin_min_betas_`x'.dta, uniqusing sort
	drop _b_cons _se_cons _merge
	
	rename _b_co_kill _b_co_kill_`x'
	rename _se_co_kill _se_co_kill_`x'
	rename _b_in_kill _b_in_kill_`x'
	rename _se_in_kill _se_in_kill_`x'
	rename _b_co_min _b_co_min_`x' 
	rename _se_co_min _se_co_min_`x'
	rename _b_in_min _b_in_min_`x'
	rename _se_in_min _se_in_min_`x'
	}

* Now we keep only one observation per strata
sort cem_strata week district
gen marker = 1 if cem_strata[_n] != cem_strata[_n+1]
drop if marker != 1
drop if cem_w>800
expand = 15

* Generate variable that identifies lag
bysort cem_strata week: gen lago = [_n]
replace lago = lago - 8

* Turn _b_* variables into one that gives coefficient and se for insurgent and coalition at each lag
local wh1 "co_kill co_min in_kill in_min"
local what "b se"
local lags1 "1 2 3 4 5 6 7"
foreach y of local what {
	foreach x of local wh1 {
		foreach z of local lags1 {
			qui replace _`y'_`x'_p = _b_`x'_f`z'p if lago == `z'
			qui replace _`y'_`x'_p = _b_`x'_l`z'p if lago == -`z'
			}
		}
	}


* Generate some additional variables for the graph
* Use loop to generate graphing variables for differences and treatment and control
local wh1 "co_kill co_min in_kill in_min"
qui foreach z of local wh1 {
	sort cem_strata lago
	gen mean_`z' = 0 
	gen se_`z' = 0 
	gen a_`z' = 0 
	gen upper_`z' = 0 
	gen lower_`z' = 0 
	
	forval x = -7/7 {
		mean _b_`z'_p if lago == `x' & cem_w < 800 [iw=cem_w]
		matrix b1=e(b) 
		matrix V1=e(V)
		scalar b1=b1[1,1] 
		scalar varb1=V1[1,1] 
		scalar list b1 varb1
	
		replace mean_`z' = b1 if lago == `x'
		replace se_`z' = sqrt(varb1) if lago == `x'
		replace a_`z' = 1.96 * se_`z' if lago == `x'
		replace upper_`z' = mean_`z' + a_`z' if lago == `x'
		replace lower_`z' = mean_`z' - a_`z' if lago == `x'
		}
	}


sort cem_strata lago

twoway (line mean_co_min lago if lago > -6 & lago < 8 & cem_strata==1 , lcolor(blue) xline(0, lcolor(black)) ) ///
	(line upper_co_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)) ///
	(line lower_co_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)), ///
	title("Marginal Effect of Coalition Killings", size(medsmall)) ///
	ytitle(SIGACTS/100000, size(medsmall)) ytitle(, linegap(.2)) ylabel(, glcolor(gs12)) ///
	xtitle(Period from treatment) xtitle(, size(medsmall)) xtitle(, margin(small)) xtick(-5(1)7) xlabel(-5(1)7) ///
	legend(col(2) order(1 2) label(1 "Weighted difference in means") size(vsmall)  ///
	label(2 "95% Confidence Interval") label(3 " ") off) name(dif_co_min, replace) 

twoway (line mean_in_min lago if lago > -6 & lago < 8 & cem_strata==1 , lcolor(blue) xline(0, lcolor(black)) ) ///
	(line upper_in_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)) ///
	(line lower_in_min lago if lago > -6 & lago < 8 & cem_strata==1, lcolor(gs5) lpattern(dash)), ///
	title("Marginal Effect of Insurgent Killings", size(medsmall)) ///
	ytitle(SIGACTS/100000, size(medsmall)) ytitle(, linegap(.2)) ylabel(, glcolor(gs12)) ///
	xtitle(Period from treatment) xtitle(, size(medsmall)) xtitle(, margin(small)) xtick(-5(1)7) xlabel(-5(1)7) ///	
	legend(col(2) order(1 2) label(1 "Weighted difference in means") size(vsmall) ///
	label(2 "95% Confidence Interval") label(3 " ") off) name(dif_in_min, replace) 
		
graph combine dif_co_min dif_in_min, ycommon colfirst cols(1) ///
	title(SE Fig. 5. Effect of Civilian Killings on Insurgent Violence in Matched Strata, size(medsmall)) ///
	subtitle((Matching on 4-period history of SIGACTS/100000, and average insurgent attacks), size(small)) imargin(small) ///
	name(cem_pooled_tightq3, replace)

	
	
	

